<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\SMSService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Mail;

class SettingsController extends Controller
{
    protected $smsService;

    public function __construct(SMSService $smsService)
    {
        $this->smsService = $smsService;
    }

    /**
     * Show settings page
     */
    public function index()
    {
        return view('admin.settings.index');
    }

    /**
     * Update settings
     */
    public function update(Request $request, $section)
    {
        switch ($section) {
            case 'general':
                return $this->updateGeneral($request);
            case 'email':
                return $this->updateEmail($request);
            case 'stripe':
                return $this->updateStripe($request);
            case 'paypal':
                return $this->updatePaypal($request);
            case 'sms':
                return $this->updateSMS($request);
            case 'push':
                return $this->updatePush($request);
            case 'google':
                return $this->updateGoogle($request);
            case 'facebook':
                return $this->updateFacebook($request);
            case 'security':
                return $this->updateSecurity($request);
            default:
                return back()->with('error', 'Invalid section');
        }
    }

    /**
     * Update general settings
     */
    protected function updateGeneral(Request $request)
    {
        $validated = $request->validate([
            'app_name' => 'required|string|max:255',
            'default_language' => 'required|in:en,ar',
            'timezone' => 'required|string',
            'currency' => 'required|string',
            'date_format' => 'required|string',
            'support_email' => 'required|email',
            'support_phone' => 'required|string',
        ]);

        // Update .env file or database settings
        $this->updateEnvFile([
            'APP_NAME' => $validated['app_name'],
            'APP_TIMEZONE' => $validated['timezone'],
            'DEFAULT_CURRENCY' => $validated['currency'],
        ]);

        Cache::forget('settings');

        return back()->with('success', 'General settings updated successfully');
    }

    /**
     * Update email settings
     */
    protected function updateEmail(Request $request)
    {
        $validated = $request->validate([
            'mail_driver' => 'required|string',
            'mail_from' => 'required|email',
            'mail_from_name' => 'required|string',
            'mail_host' => 'nullable|string',
            'mail_port' => 'nullable|integer',
            'mail_username' => 'nullable|string',
            'mail_password' => 'nullable|string',
            'mail_encryption' => 'nullable|boolean',
        ]);

        $this->updateEnvFile([
            'MAIL_MAILER' => $validated['mail_driver'],
            'MAIL_FROM_ADDRESS' => $validated['mail_from'],
            'MAIL_FROM_NAME' => $validated['mail_from_name'],
            'MAIL_HOST' => $validated['mail_host'],
            'MAIL_PORT' => $validated['mail_port'],
            'MAIL_USERNAME' => $validated['mail_username'],
            'MAIL_PASSWORD' => $validated['mail_password'],
            'MAIL_ENCRYPTION' => $validated['mail_encryption'] ? 'tls' : null,
        ]);

        return back()->with('success', 'Email settings updated successfully');
    }

    /**
     * Update Stripe settings
     */
    protected function updateStripe(Request $request)
    {
        $validated = $request->validate([
            'stripe_enabled' => 'boolean',
            'stripe_key' => 'nullable|string',
            'stripe_secret' => 'nullable|string',
            'stripe_webhook_secret' => 'nullable|string',
        ]);

        $this->updateEnvFile([
            'STRIPE_KEY' => $validated['stripe_key'],
            'STRIPE_SECRET' => $validated['stripe_secret'],
            'STRIPE_WEBHOOK_SECRET' => $validated['stripe_webhook_secret'],
        ]);

        return back()->with('success', 'Stripe settings updated successfully');
    }

    /**
     * Update PayPal settings
     */
    protected function updatePaypal(Request $request)
    {
        $validated = $request->validate([
            'paypal_enabled' => 'boolean',
            'paypal_mode' => 'required|in:sandbox,live',
            'paypal_client_id' => 'nullable|string',
            'paypal_secret' => 'nullable|string',
        ]);

        $this->updateEnvFile([
            'PAYPAL_MODE' => $validated['paypal_mode'],
            'PAYPAL_CLIENT_ID' => $validated['paypal_client_id'],
            'PAYPAL_CLIENT_SECRET' => $validated['paypal_secret'],
        ]);

        return back()->with('success', 'PayPal settings updated successfully');
    }

    /**
     * Update SMS settings
     */
    protected function updateSMS(Request $request)
    {
        $validated = $request->validate([
            'sms_enabled' => 'boolean',
            'twilio_sid' => 'nullable|string',
            'twilio_token' => 'nullable|string',
            'twilio_from' => 'nullable|string',
        ]);

        $this->updateEnvFile([
            'TWILIO_SID' => $validated['twilio_sid'],
            'TWILIO_TOKEN' => $validated['twilio_token'],
            'TWILIO_FROM' => $validated['twilio_from'],
        ]);

        return back()->with('success', 'SMS settings updated successfully');
    }

    /**
     * Update Push notification settings
     */
    protected function updatePush(Request $request)
    {
        $validated = $request->validate([
            'push_enabled' => 'boolean',
            'fcm_server_key' => 'nullable|string',
            'fcm_sender_id' => 'nullable|string',
        ]);

        $this->updateEnvFile([
            'FCM_SERVER_KEY' => $validated['fcm_server_key'],
            'FCM_SENDER_ID' => $validated['fcm_sender_id'],
        ]);

        return back()->with('success', 'Push notification settings updated successfully');
    }

    /**
     * Update Google settings
     */
    protected function updateGoogle(Request $request)
    {
        $validated = $request->validate([
            'google_enabled' => 'boolean',
            'google_client_id' => 'nullable|string',
            'google_client_secret' => 'nullable|string',
        ]);

        $this->updateEnvFile([
            'GOOGLE_CLIENT_ID' => $validated['google_client_id'],
            'GOOGLE_CLIENT_SECRET' => $validated['google_client_secret'],
        ]);

        return back()->with('success', 'Google settings updated successfully');
    }

    /**
     * Update Facebook settings
     */
    protected function updateFacebook(Request $request)
    {
        $validated = $request->validate([
            'facebook_enabled' => 'boolean',
            'facebook_app_id' => 'nullable|string',
            'facebook_app_secret' => 'nullable|string',
        ]);

        $this->updateEnvFile([
            'FACEBOOK_CLIENT_ID' => $validated['facebook_app_id'],
            'FACEBOOK_CLIENT_SECRET' => $validated['facebook_app_secret'],
        ]);

        return back()->with('success', 'Facebook settings updated successfully');
    }

    /**
     * Update security settings
     */
    protected function updateSecurity(Request $request)
    {
        $validated = $request->validate([
            'force_https' => 'boolean',
            'two_factor_enabled' => 'boolean',
            'session_timeout' => 'required|integer|min:5',
            'max_login_attempts' => 'required|integer|min:3',
            'lockout_duration' => 'required|integer|min:1',
        ]);

        $this->updateEnvFile([
            'SESSION_LIFETIME' => $validated['session_timeout'],
        ]);

        // Store other settings in cache or database
        Cache::put('security_settings', $validated, now()->addYears(1));

        return back()->with('success', 'Security settings updated successfully');
    }

    /**
     * Toggle maintenance mode
     */
    public function maintenance(Request $request)
    {
        $validated = $request->validate([
            'maintenance_mode' => 'required|boolean',
            'maintenance_message' => 'nullable|string',
            'expected_return' => 'nullable|date',
        ]);

        if ($validated['maintenance_mode']) {
            Artisan::call('down', [
                '--message' => $validated['maintenance_message'] ?? 'System maintenance',
                '--retry' => 60,
            ]);
        } else {
            Artisan::call('up');
        }

        return back()->with('success', 'Maintenance mode updated successfully');
    }

    /**
     * Test email
     */
    public function testEmail(Request $request)
    {
        $email = $request->email;

        try {
            Mail::raw('This is a test email from Privasee Admin Panel.', function ($message) use ($email) {
                $message->to($email)
                    ->subject('Test Email from Privasee');
            });

            return response()->json([
                'success' => true,
                'message' => 'Test email sent successfully to ' . $email
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send email: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test SMS
     */
    public function testSMS(Request $request)
    {
        $phone = $request->phone;

        try {
            $result = $this->smsService->send($phone, 'This is a test SMS from Privasee Admin Panel.');

            return response()->json($result);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send SMS: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update .env file
     */
    protected function updateEnvFile(array $data)
    {
        $envFile = base_path('.env');
        $env = file_get_contents($envFile);

        foreach ($data as $key => $value) {
            if ($value === null) continue;

            $value = is_bool($value) ? ($value ? 'true' : 'false') : $value;

            // Check if key exists
            if (preg_match("/^{$key}=.*/m", $env)) {
                $env = preg_replace("/^{$key}=.*/m", "{$key}={$value}", $env);
            } else {
                $env .= "\n{$key}={$value}";
            }
        }

        file_put_contents($envFile, $env);
    }
}
