<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;

class SubscriptionPlanController extends Controller
{
    /**
     * Display subscription plans listing
     */
    public function index(Request $request)
    {
        $type = $request->get('type', 'business'); // Default to business plans

        $plans = SubscriptionPlan::where('type', $type)
            ->withCount(['subscriptions', 'activeSubscriptions'])
            ->orderBy('sort_order')
            ->get();

        $stats = [
            'total_plans' => SubscriptionPlan::where('type', $type)->count(),
            'active_plans' => SubscriptionPlan::where('type', $type)->where('is_active', true)->count(),
            'total_subscriptions' => \App\Models\Subscription::whereHas('plan', function($q) use ($type) {
                $q->where('type', $type);
            })->count(),
            'monthly_revenue' => \App\Models\Payment::where('status', 'completed')
                ->where('type', 'subscription')
                ->whereMonth('created_at', now()->month)
                ->sum('amount'),
        ];

        // Separate counts for business, user, and marketing plans
        $businessPlansCount = SubscriptionPlan::where('type', 'business')->count();
        $userPlansCount = SubscriptionPlan::where('type', 'user')->count();
        $marketingPlansCount = SubscriptionPlan::where('type', 'marketing')->count();

        return view('admin.subscription-plans.index', compact('plans', 'stats', 'type', 'businessPlansCount', 'userPlansCount', 'marketingPlansCount'));
    }

    /**
     * Show create form
     */
    public function create()
    {
        return view('admin.subscription-plans.create');
    }

    /**
     * Store new subscription plan
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'type' => 'required|in:business,user',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'duration_type' => 'required|in:monthly,yearly,custom',
            'features' => 'required|array',
            'features.*' => 'string',
            'max_venues' => 'nullable|integer|min:1',
            'max_offers' => 'nullable|integer|min:1',
            'max_staff' => 'nullable|integer|min:1',
            'max_photos_per_venue' => 'nullable|integer|min:1',
            'analytics_access' => 'boolean',
            'priority_support' => 'boolean',
            'featured_listing' => 'boolean',
            'custom_branding' => 'boolean',
            'api_access' => 'boolean',
            'sort_order' => 'nullable|integer',
            'is_active' => 'boolean',
            'is_popular' => 'boolean',
        ]);

        // Convert features array to JSON
        $validated['features'] = json_encode($validated['features']);

        $plan = SubscriptionPlan::create($validated);

        return response()->json([
            'success' => true,
            'message' => ucfirst($validated['type']) . ' subscription plan created successfully',
            'data' => $plan
        ]);
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);
        return response()->json([
            'success' => true,
            'plan' => $plan
        ]);
    }

    /**
     * Update subscription plan
     */
    public function update(Request $request, $id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'duration_type' => 'required|in:monthly,yearly,custom',
            'features' => 'required|array',
            'features.*' => 'string',
            'max_venues' => 'nullable|integer|min:1',
            'max_offers' => 'nullable|integer|min:1',
            'max_staff' => 'nullable|integer|min:1',
            'max_photos_per_venue' => 'nullable|integer|min:1',
            'analytics_access' => 'boolean',
            'priority_support' => 'boolean',
            'featured_listing' => 'boolean',
            'custom_branding' => 'boolean',
            'api_access' => 'boolean',
            'sort_order' => 'nullable|integer',
            'is_active' => 'boolean',
            'is_popular' => 'boolean',
        ]);

        // Convert features array to JSON
        $validated['features'] = json_encode($validated['features']);

        $plan->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Subscription plan updated successfully',
            'data' => $plan
        ]);
    }

    /**
     * Toggle plan status
     */
    public function toggle($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);
        $plan->update(['is_active' => !$plan->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'Plan status updated',
            'data' => $plan
        ]);
    }

    /**
     * Toggle popular badge
     */
    public function togglePopular($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);
        $plan->update(['is_popular' => !$plan->is_popular]);

        return response()->json([
            'success' => true,
            'message' => 'Popular badge updated',
            'data' => $plan
        ]);
    }

    /**
     * Delete subscription plan
     */
    public function destroy($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        // Check if plan has active subscriptions
        if ($plan->subscriptions()->where('status', 'active')->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete plan with active subscriptions'
            ], 400);
        }

        $plan->delete();

        return response()->json([
            'success' => true,
            'message' => 'Subscription plan deleted successfully'
        ]);
    }

    /**
     * View plan subscribers
     */
    public function subscribers($id)
    {
        $plan = SubscriptionPlan::findOrFail($id);

        $subscriptions = $plan->subscriptions()
            ->with(['business', 'business.owner'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.subscription-plans.subscribers', compact('plan', 'subscriptions'));
    }
}
