<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\BusinessStaff;
use App\Models\Offer;
use App\Models\Redemption;
use App\Models\Review;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
// Case 1: Business Owner
        $business = $user->businesses()
            ->with(['venues', 'subscriptionPlan'])
            ->first();

        // Case 2: Staff User (business account but not the owner)
        if (!$business && $user->account_type === 'business') {
            $staffRecord = BusinessStaff::where('user_id', $user->id)
                ->where('status', 'active') // ensures accepted team member
                ->first();

            if ($staffRecord) {
                $business = Business::with(['venues', 'subscriptionPlan'])
                    ->where('id', $staffRecord->business_id)
                    ->first();
            }
        }

        if (!$business) {
            return redirect()->route('business.register')->with('error', 'Please register your business first');
        }

        // Overview Stats
        $stats = [
            'total_venues' => $business->venues()->count(),
            'active_venues' => $business->venues()->where('status', 'active')->count(),
            'total_offers' => $business->offers()->count(),
            'active_offers' => $business->offers()
                ->where('status', 'active')
                ->where('start_date', '<=', now())
                ->where('end_date', '>=', now())
                ->count(),
            'pending_approval' => $business->offers()
                ->where('approval_status', 'pending')
                ->count(),
            'total_redemptions' => Redemption::whereHas('offer', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->count(),
            'pending_redemptions' => Redemption::whereHas('offer', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->where('status', 'pending')->count(),
            'total_reviews' => $business->total_reviews,
            'average_rating' => $business->rating,
            'followers_count' => $business->followers()->count(),
        ];

        // Recent Activity
        $recentRedemptions = Redemption::with(['user', 'offer', 'venue'])
            ->whereHas('offer', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        $recentReviews = Review::with(['user', 'venue'])
            ->whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        // Performance Metrics (Last 30 Days)
        $thirtyDaysAgo = now()->subDays(30);

        $performance = [
            'redemptions_30d' => Redemption::whereHas('offer', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->where('created_at', '>=', $thirtyDaysAgo)->count(),

            'views_30d' => $business->venues()->sum('view_count'),

            'new_followers_30d' => $business->followers()
                ->where('created_at', '>=', $thirtyDaysAgo)
                ->count(),

            'reviews_30d' => Review::whereHas('venue', function($q) use ($business) {
                $q->where('business_id', $business->id);
            })->where('created_at', '>=', $thirtyDaysAgo)->count(),
        ];

        // Top Performing Offers
        $topOffers = Offer::where('business_id', $business->id)
            ->where('status', 'active')
            ->orderBy('redemption_count', 'desc')
            ->orderBy('view_count', 'desc')
            ->take(5)
            ->get();

        // Subscription Info
        $subscription = [
            'plan' => $business->subscriptionPlan,
            'status' => $business->subscription_status,
            'start_date' => $business->subscription_start_date,
            'end_date' => $business->subscription_end_date,
            'days_remaining' => $business->subscription_end_date
                ? now()->diffInDays($business->subscription_end_date)
                : null,
            'is_active' => $business->hasActiveSubscription(),
        ];

        return view('business.dashboard', compact(
            'business',
            'stats',
            'recentRedemptions',
            'recentReviews',
            'performance',
            'topOffers',
            'subscription'
        ));
    }

    public function analytics(Request $request)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business found'
            ], 404);
        }

        $days = $request->days ?? 30;
        $startDate = now()->subDays($days);

        // Daily redemptions
        $dailyRedemptions = Redemption::whereHas('offer', function($q) use ($business) {
            $q->where('business_id', $business->id);
        })
            ->where('created_at', '>=', $startDate)
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Offers performance
        $offersPerformance = Offer::where('business_id', $business->id)
            ->withCount('redemptions')
            ->orderBy('redemptions_count', 'desc')
            ->take(10)
            ->get();

        // Venues performance
        $venuesPerformance = $business->venues()
            ->withCount('redemptions')
            ->orderBy('redemptions_count', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'daily_redemptions' => $dailyRedemptions,
                'offers_performance' => $offersPerformance,
                'venues_performance' => $venuesPerformance,
            ]
        ]);
    }
}
