<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Mail\TeamInvitation;
use App\Models\Business;
use App\Models\BusinessStaff;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class TeamController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $business = $user->business;

        if (!$business) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No business found'
                ], 404);
            }
            return redirect()->route('business.dashboard')
                ->with('error', 'Please complete your business registration first.');
        }

        // Check subscription status
        $hasActiveSubscription = $business->activeSubscription()->exists();

        if (!$hasActiveSubscription) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please subscribe to a plan to access team management',
                    'requires_subscription' => true
                ], 403);
            }
            return view('business.team.index', [
                'requires_subscription' => true,
                'business' => $business,
                'staff' => collect(),
                'stats' => []
            ]);
        }

        // Check permission
        $staffMember = BusinessStaff::where('business_id', $business->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$staffMember || !$staffMember->canManageStaff()) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to manage staff'
                ], 403);
            }
            return view('business.team.index', [
                'no_permission' => true,
                'business' => $business,
                'staff' => collect(),
                'stats' => []
            ]);
        }

        $query = BusinessStaff::with('user')
            ->where('business_id', $business->id);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by role
        if ($request->has('role')) {
            $query->where('role', $request->role);
        }

        // Search by name or email
        if ($request->has('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $staff = $query->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 15);

        $stats = [
            'total' => BusinessStaff::where('business_id', $business->id)->count(),
            'active' => BusinessStaff::where('business_id', $business->id)
                ->where('status', 'active')
                ->count(),
            'pending' => BusinessStaff::where('business_id', $business->id)
                ->where('status', 'pending_invitation')
                ->count(),
            'max_limit' => $business->subscriptionPlan->max_staff ?? 'Unlimited',
        ];

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $staff
            ]);
        }

        return view('business.team.index', compact('staff', 'business', 'stats'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return redirect()->route('business.dashboard')
                ->with('error', 'Please complete your business registration first.');
        }

        // Check permission
        $staffMember = BusinessStaff::where('business_id', $business->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$staffMember || !$staffMember->canManageStaff()) {
            return redirect()->route('business.team.index')
                ->with('error', 'You do not have permission to manage staff.');
        }

        // Validate input
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'role' => 'required|in:manager,staff',
            'permissions' => 'nullable|array',
            'permissions.*' => 'in:manage_venues,manage_offers,manage_staff,view_analytics,redeem_offers',
        ]);

        if ($validator->fails()) {
            return redirect()->route('business.team.index')
                ->withErrors($validator)
                ->withInput();
        }

        // Subscription staff count limit
        if ($business->subscriptionPlan) {
            $currentStaffCount = BusinessStaff::where('business_id', $business->id)
                ->where('status', '!=', 'inactive')
                ->count();

            if ($business->subscriptionPlan->max_staff &&
                $currentStaffCount >= $business->subscriptionPlan->max_staff) {

                return redirect()->route('business.team.index')
                    ->with('error', 'You have reached the maximum number of team members for your subscription plan.');
            }
        }

        // Find or create user
        $invitedUser = User::firstOrCreate(
            ['email' => $request->email],
            [
                'first_name' => explode('@', $request->email)[0],
                'last_name' => '',
                'account_type' => 'business',
                'status' => 'pending_verification',
                'password' => bcrypt(Str::random(16)),
            ]
        );

        // Check for soft-deleted or active staff
        $existingStaff = BusinessStaff::withTrashed()
            ->where('business_id', $business->id)
            ->where('user_id', $invitedUser->id)
            ->first();

        /**
         * CASE 1: Staff exists (soft-deleted → restore)
         */
        if ($existingStaff) {

            // If soft-deleted, restore and update
            if ($existingStaff->trashed()) {

                $existingStaff->restore();

                $existingStaff->update([
                    'role' => $request->role,
                    'permissions' => $request->permissions,
                    'status' => 'pending_invitation',
                    'invited_at' => now(),
                ]);

                // Send invitation again
                Mail::to($invitedUser->email)->send(new TeamInvitation($business, $existingStaff));

                return redirect()->route('business.team.index')
                    ->with('success', 'Invitation resent and team member restored.');
            }

            // If not soft-deleted → already active
            return redirect()->route('business.team.index')
                ->with('error', 'This user is already a team member or has a pending invitation.');
        }

        /**
         * CASE 2: New staff invitation
         */
        DB::beginTransaction();

        try {
            $staff = BusinessStaff::create([
                'business_id' => $business->id,
                'user_id' => $invitedUser->id,
                'role' => $request->role,
                'permissions' => $request->permissions,
                'status' => 'pending_invitation',
                'invited_at' => now(),
            ]);

            $business->increment('total_staff');

            DB::commit();

            // Send invitation email
            Mail::to($invitedUser->email)->send(new TeamInvitation($business, $staff));

            return redirect()->route('business.team.index')
                ->with('success', 'Team member invited successfully.');

        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->route('business.team.index')
                ->with('error', 'Failed to invite team member. Please try again.');
        }
    }


    public function show($id)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business found'
            ], 404);
        }

        $staff = BusinessStaff::with('user')
            ->where('business_id', $business->id)
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $staff
        ]);
    }

    public function update(Request $request, $id)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business found'
            ], 404);
        }

        // Check permission
        $staffMember = BusinessStaff::where('business_id', $business->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$staffMember || !$staffMember->canManageStaff()) {
            return response()->json([
                'success' => false,
                'message' => 'You do not have permission to manage staff'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'role' => 'sometimes|required|in:manager,staff',
            'permissions' => 'nullable|array',
            'permissions.*' => 'in:manage_venues,manage_offers,manage_staff,view_analytics,redeem_offers',
            'status' => 'sometimes|required|in:active,inactive',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $staff = BusinessStaff::where('business_id', $business->id)
            ->findOrFail($id);

        // Cannot modify owner
        if ($staff->isOwner()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot modify owner role'
            ], 400);
        }

        $staff->update($request->only(['role', 'permissions', 'status']));

        return response()->json([
            'success' => true,
            'message' => 'Team member updated successfully',
            'data' => $staff->fresh()->load('user')
        ]);
    }

    public function destroy($id)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business found'
            ], 404);
        }

        // Check permission
        $staffMember = BusinessStaff::where('business_id', $business->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$staffMember || !$staffMember->canManageStaff()) {
            return response()->json([
                'success' => false,
                'message' => 'You do not have permission to manage staff'
            ], 403);
        }

        $staff = BusinessStaff::where('business_id', $business->id)
            ->findOrFail($id);

        // Cannot remove owner
        if ($staff->isOwner()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot remove business owner'
            ], 400);
        }

        // Cannot remove yourself if you're the only manager
        if ($staff->user_id === $user->id) {
            $managerCount = BusinessStaff::where('business_id', $business->id)
                ->where('role', 'manager')
                ->where('status', 'active')
                ->count();

            if ($managerCount <= 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot remove yourself as the only manager'
                ], 400);
            }
        }

        DB::beginTransaction();
        try {
            $staff->delete();

            // Decrement business staff count
            $business->decrement('total_staff');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Team member removed successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove team member'
            ], 500);
        }
    }

    public function acceptInvitation(Request $request)
    {
        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'invitation_id' => 'required|exists:business_staff,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $invitation = BusinessStaff::where('id', $request->invitation_id)
            ->where('user_id', $user->id)
            ->where('status', 'pending_invitation')
            ->first();

        if (!$invitation) {
            return response()->json([
                'success' => false,
                'message' => 'Invitation not found or already accepted'
            ], 404);
        }

        $invitation->update([
            'status' => 'active',
            'joined_at' => now(),
        ]);

        // Update user account type if needed
        if ($user->account_type !== 'business') {
            $user->update(['account_type' => 'business']);
        }

        return response()->json([
            'success' => true,
            'message' => 'Invitation accepted successfully',
            'data' => [
                'business' => $invitation->business,
                'staff' => $invitation,
            ]
        ]);
    }

    public function rejectInvitation(Request $request)
    {
        $user = auth()->user();

        $validator = Validator::make($request->all(), [
            'invitation_id' => 'required|exists:business_staff,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'errors' => $validator->errors()
            ], 422);
        }

        $invitation = BusinessStaff::where('id', $request->invitation_id)
            ->where('user_id', $user->id)
            ->where('status', 'pending_invitation')
            ->first();

        if (!$invitation) {
            return response()->json([
                'success' => false,
                'message' => 'Invitation not found'
            ], 404);
        }

        DB::beginTransaction();
        try {
            $businessId = $invitation->business_id;
            $invitation->delete();

            // Decrement business staff count
            Business::find($businessId)->decrement('total_staff');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invitation rejected'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to reject invitation'
            ], 500);
        }
    }

    public function myInvitations()
    {
        $user = auth()->user();

        $invitations = BusinessStaff::with('business')
            ->where('user_id', $user->id)
            ->where('status', 'pending_invitation')
            ->orderBy('invited_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $invitations
        ]);
    }

    public function resendInvitation($id)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business found'
            ], 404);
        }

        $staff = BusinessStaff::with('user')
            ->where('business_id', $business->id)
            ->where('status', 'pending_invitation')
            ->findOrFail($id);

        // TODO: Resend invitation email
         Mail::to($staff->user->email)->send(new TeamInvitation($business, $staff));

        $staff->update(['invited_at' => now()]);

        return redirect()->route('business.team.index')
            ->with('success', 'Invitation resent successfully.');
    }

    public function stats()
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business found'
            ], 404);
        }

        $stats = [
            'total_staff' => BusinessStaff::where('business_id', $business->id)->count(),
            'active_staff' => BusinessStaff::where('business_id', $business->id)
                ->where('status', 'active')
                ->count(),
            'pending_invitations' => BusinessStaff::where('business_id', $business->id)
                ->where('status', 'pending_invitation')
                ->count(),
            'managers' => BusinessStaff::where('business_id', $business->id)
                ->where('role', 'manager')
                ->where('status', 'active')
                ->count(),
            'staff_members' => BusinessStaff::where('business_id', $business->id)
                ->where('role', 'staff')
                ->where('status', 'active')
                ->count(),
            'max_staff_limit' => $business->subscriptionPlan->max_staff ?? null,
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

// Store or update role with dynamic permissions
    public function storeRole(Request $request)
    {
        $request->validate([
            'role_name' => 'required|string',
            'permissions' => 'required|array',
        ]);
        $user = auth()->user();
        $business = $user->businesses()->first();
        $businessId = $business->id;

        $role = BusinessStaff::updateOrCreate(
            [
                'user_id' => $user->id,
                'business_id' => $businessId,
                'role' => $request->role_name,
            ],
            [
                'permissions' => $request->permissions,
                'status' => 'active',
            ]
        );

        return redirect()->back()->with('success', 'Role saved successfully.');
    }

    public function accept($id)
    {
        $invitation = BusinessStaff::with(['user', 'business'])
            ->where('id', $id)
            ->where('status', 'pending_invitation')
            ->first();

        if (!$invitation) {
            return view('business.team.invitation-invalid'); // Show invalid/expired link
        }

        // If user is not verified or has no password
        $user = $invitation->user;

        return view('business.team.accept-invitation', compact('invitation', 'user'));
    }

    public function complete(Request $request, $id)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'password' => 'required|string|min:8|confirmed',

        ]);

        $staff = BusinessStaff::with('user')
            ->where('id', $id)
            ->where('status', 'pending_invitation')
            ->firstOrFail();

        $user = $staff->user;
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->password = bcrypt($request->password);
        $user->status = 'active';
        $user->email_verified_at = now();
        $user->save();

        $staff->status = 'active';
        $staff->joined_at = now();
        $staff->save();

        // Log the user in
        auth()->login($user);

        return redirect()->route('business.dashboard')
            ->with('success', 'Invitation accepted successfully. Welcome to the team!');
    }


}
