<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class GdprRequest extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'user_id',
        'type',
        'status',
        'reason',
        'admin_notes',
        'processed_by',
        'processed_at',
        'export_file_path',
        'expires_at',
    ];

    protected $casts = [
        'processed_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    /**
     * Get the user who made the request
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the admin who processed the request
     */
    public function processor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'processed_by');
    }

    /**
     * Check if request is pending
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if request is completed
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Mark as processing
     */
    public function markAsProcessing(): void
    {
        $this->update([
            'status' => 'processing',
            'processed_by' => auth()->id(),
        ]);
    }

    /**
     * Mark as completed
     */
    public function markAsCompleted(string $filePath = null): void
    {
        $this->update([
            'status' => 'completed',
            'processed_at' => now(),
            'processed_by' => auth()->id(),
            'export_file_path' => $filePath,
            'expires_at' => now()->addDays(30),
        ]);
    }

    /**
     * Mark as rejected
     */
    public function markAsRejected(string $reason): void
    {
        $this->update([
            'status' => 'rejected',
            'processed_at' => now(),
            'processed_by' => auth()->id(),
            'admin_notes' => $reason,
        ]);
    }

    /**
     * Check if export file is expired
     */
    public function isExpired(): bool
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    /**
     * Get pending requests
     */
    public static function getPending()
    {
        return self::where('status', 'pending')
            ->with('user')
            ->orderBy('created_at', 'asc');
    }
}
