<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Otp extends Model
{
    use HasFactory;

    protected $fillable = [
        'phone',
        'email',
        'code',
        'type',
        'expires_at',
        'verified_at',
        'attempts',
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'verified_at' => 'datetime',
    ];

    /**
     * Check if OTP is expired
     */
    public function isExpired()
    {
        return $this->expires_at->isPast();
    }

    /**
     * Check if OTP is verified
     */
    public function isVerified()
    {
        return !is_null($this->verified_at);
    }

    /**
     * Check if max attempts reached
     */
    public function maxAttemptsReached()
    {
        return $this->attempts >= 5;
    }

    /**
     * Mark as verified
     */
    public function markAsVerified()
    {
        $this->update([
            'verified_at' => now()
        ]);
    }

    /**
     * Increment attempts
     */
    public function incrementAttempts()
    {
        $this->increment('attempts');
    }

    /**
     * Create new OTP
     */
    public static function createNew($identifier, $code, $type = 'login', $expiresInMinutes = 10)
    {
        // Delete old unverified OTPs
        if (filter_var($identifier, FILTER_VALIDATE_EMAIL)) {
            static::where('email', $identifier)
                ->whereNull('verified_at')
                ->delete();
        } else {
            static::where('phone', $identifier)
                ->whereNull('verified_at')
                ->delete();
        }

        // Create new OTP
        return static::create([
            filter_var($identifier, FILTER_VALIDATE_EMAIL) ? 'email' : 'phone' => $identifier,
            'code' => $code,
            'type' => $type,
            'expires_at' => Carbon::now()->addMinutes($expiresInMinutes),
            'attempts' => 0,
        ]);
    }

    /**
     * Verify OTP code
     */
    public static function verify($identifier, $code, $type = 'login')
    {
        $query = static::where('code', $code)
            ->where('type', $type)
            ->whereNull('verified_at');

        if (filter_var($identifier, FILTER_VALIDATE_EMAIL)) {
            $query->where('email', $identifier);
        } else {
            $query->where('phone', $identifier);
        }

        $otp = $query->first();

        if (!$otp) {
            return [
                'success' => false,
                'message' => 'Invalid OTP code'
            ];
        }

        if ($otp->isExpired()) {
            return [
                'success' => false,
                'message' => 'OTP code has expired'
            ];
        }

        if ($otp->maxAttemptsReached()) {
            return [
                'success' => false,
                'message' => 'Maximum verification attempts reached. Request a new code.'
            ];
        }

        $otp->markAsVerified();

        return [
            'success' => true,
            'message' => 'OTP verified successfully',
            'otp' => $otp
        ];
    }
}
