<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Role extends Model
{
    protected $fillable = [
        'name',
        'display_name',
        'description',
        'type',
        'is_system',
    ];

    protected $casts = [
        'is_system' => 'boolean',
    ];

    /**
     * Permissions for this role
     */
    public function permissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'role_permission')
            ->withTimestamps();
    }

    /**
     * Users with this role
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'user_role')
            ->withPivot('business_id')
            ->withTimestamps();
    }

    /**
     * Check if role has permission
     */
    public function hasPermission(string $permissionName): bool
    {
        // Use eager loaded permissions if available
        if ($this->relationLoaded('permissions')) {
            return $this->permissions->where('name', $permissionName)->isNotEmpty();
        }

        // Fallback to query
        return $this->permissions()
            ->where('name', $permissionName)
            ->exists();
    }

    /**
     * Give permission to role
     */
    public function givePermission($permission)
    {
        if (is_string($permission)) {
            $permission = Permission::where('name', $permission)->firstOrFail();
        }

        if (!$this->hasPermission($permission->name)) {
            $this->permissions()->attach($permission->id);
        }

        return $this;
    }

    /**
     * Revoke permission from role
     */
    public function revokePermission($permission)
    {
        if (is_string($permission)) {
            $permission = Permission::where('name', $permission)->firstOrFail();
        }

        $this->permissions()->detach($permission->id);

        return $this;
    }

    /**
     * Sync permissions
     */
    public function syncPermissions(array $permissionIds)
    {
        $this->permissions()->sync($permissionIds);

        return $this;
    }

    /**
     * Check if system role (cannot be deleted)
     */
    public function isSystemRole(): bool
    {
        return $this->is_system;
    }

    /**
     * Get roles by type
     */
    public static function byType(string $type)
    {
        return self::where('type', $type)->get();
    }

    /**
     * Get admin roles
     */
    public static function adminRoles()
    {
        return self::byType('admin');
    }

    /**
     * Get business roles
     */
    public static function businessRoles()
    {
        return self::byType('business');
    }
}
