<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, HasRoles, SoftDeletes;

    protected $fillable = [
        'first_name',
        'last_name',
        'email',
        'phone',
        'password',
        'gender',
        'date_of_birth',
        'profile_image',
        'account_type',
        'status',
        'language',
        'dark_mode',
        'interests',
        'preferences',
        'referral_code',
        'referred_by',
        'biometric_enabled',
        'device_token',
        'fcm_token',
        'last_active_at',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'phone_verified_at' => 'datetime',
        'password' => 'hashed',
        'date_of_birth' => 'date',
        'dark_mode' => 'boolean',
        'biometric_enabled' => 'boolean',
        'interests' => 'array',
        'preferences' => 'array',
        'last_active_at' => 'datetime',
    ];

    /**
     * Eager load relationships
     */
    protected $with = ['roles.permissions'];

    // Relationships
    public function notifications()
    {
        return $this->morphMany(Notification::class, 'notifiable')->orderBy('created_at', 'desc');
    }

    public function unreadNotifications()
    {
        return $this->morphMany(Notification::class, 'notifiable')->whereNull('read_at')->orderBy('created_at', 'desc');
    }

    public function socialAccounts()
    {
        return $this->hasMany(SocialAccount::class);
    }

    public function referrer()
    {
        return $this->belongsTo(User::class, 'referred_by');
    }

    public function referrals()
    {
        return $this->hasMany(User::class, 'referred_by');
    }

    public function businesses()
    {
        return $this->hasMany(Business::class, 'owner_id');
    }

    public function business()
    {
        return $this->hasOne(Business::class, 'owner_id')->latestOfMany();
    }

    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    public function activeSubscription()
    {
        return $this->hasOne(Subscription::class)->where('status', 'active')->latest();
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function redemptions()
    {
        return $this->hasMany(Redemption::class);
    }

    public function spotlights()
    {
        return $this->hasMany(Spotlight::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function favorites()
    {
        return $this->hasMany(Favorite::class);
    }

    public function follows()
    {
        return $this->hasMany(Follow::class);
    }

    public function spotlightLikes()
    {
        return $this->hasMany(SpotlightLike::class);
    }

    public function spotlightComments()
    {
        return $this->hasMany(SpotlightComment::class);
    }

    public function businessStaff()
    {
        return $this->hasMany(BusinessStaff::class);
    }

    public function activeBusinessMemberships()
    {
        return $this->hasMany(BusinessStaff::class)->where('status', 'active');
    }

    public function pendingBusinessInvitations()
    {
        return $this->hasMany(BusinessStaff::class)->where('status', 'pending_invitation');
    }

    /**
     * Roles relationship
     */
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class, 'user_role')
            ->withPivot('business_id')
            ->withTimestamps();
    }

    /**
     * Check if user has role
     */
    public function hasRole($role, $businessId = null): bool
    {
        if (is_string($role)) {
            $query = $this->roles()->where('name', $role);
        } else {
            $query = $this->roles()->where('id', $role);
        }

        if ($businessId) {
            $query->wherePivot('business_id', $businessId);
        }

        return $query->exists();
    }

    /**
     * Check if user has any role
     */
    public function hasAnyRole(array $roles, $businessId = null): bool
    {
        foreach ($roles as $role) {
            if ($this->hasRole($role, $businessId)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check if user has permission
     */
    public function hasPermission(string $permission, $businessId = null): bool
    {
        // Get roles with permissions eager loaded
        if ($businessId) {
            $roles = $this->roles()->wherePivot('business_id', $businessId)->with('permissions')->get();
        } else {
            // Use eager loaded roles or load them if not already loaded
            $roles = $this->relationLoaded('roles') ? $this->roles : $this->roles()->with('permissions')->get();
        }

        foreach ($roles as $role) {
            // Check if role has the permission
            if ($role->relationLoaded('permissions')) {
                // Use eager loaded permissions
                if ($role->permissions->where('name', $permission)->isNotEmpty()) {
                    return true;
                }
            } else {
                // Fallback to query
                if ($role->hasPermission($permission)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Check if user has any permission
     */
    public function hasAnyPermission(array $permissions, $businessId = null): bool
    {
        foreach ($permissions as $permission) {
            if ($this->hasPermission($permission, $businessId)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Assign role to user
     */
    public function assignRole($role, $businessId = null)
    {
        if (is_string($role)) {
            $role = Role::where('name', $role)->firstOrFail();
        }

        if (!$this->hasRole($role->id, $businessId)) {
            $this->roles()->attach($role->id, ['business_id' => $businessId]);
        }

        return $this;
    }

    /**
     * Remove role from user
     */
    public function removeRole($role, $businessId = null)
    {
        if (is_string($role)) {
            $role = Role::where('name', $role)->firstOrFail();
        }

        if ($businessId) {
            $this->roles()
                ->wherePivot('business_id', $businessId)
                ->detach($role->id);
        } else {
            $this->roles()->detach($role->id);
        }

        return $this;
    }

    /**
     * Sync roles for user
     */
    public function syncRoles(array $roleIds, $businessId = null)
    {
        if ($businessId) {
            // Remove old roles for this business
            $this->roles()->wherePivot('business_id', $businessId)->detach();

            // Attach new roles
            foreach ($roleIds as $roleId) {
                $this->roles()->attach($roleId, ['business_id' => $businessId]);
            }
        } else {
            $this->roles()->sync($roleIds);
        }

        return $this;
    }

    /**
     * Check if super admin
     */
    public function isSuperAdmin(): bool
    {
        return $this->hasRole('super_admin');
    }

    /**
     * Check if has admin role
     */
    public function isAdminUser(): bool
    {
        return $this->hasAnyRole(['super_admin', 'admin']);
    }

    /**
     * Check if business owner
     */
    public function isBusinessOwner($businessId = null): bool
    {
        return $this->hasRole('business_owner', $businessId);
    }

    // Helpers
    public function getFullNameAttribute()
    {
        return "{$this->first_name} {$this->last_name}";
    }

    public function hasActiveSubscription()
    {
        return $this->activeSubscription()->exists();
    }

    public function isAdmin()
    {
        return $this->account_type === 'admin';
    }

    public function isBusiness()
    {
        return $this->account_type === 'business';
    }

    public function isUser()
    {
        return $this->account_type === 'user';
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($user) {
            if (empty($user->referral_code)) {
                $user->referral_code = strtoupper(substr(md5(uniqid()), 0, 8));
            }
        });
    }
}
