<?php

namespace App\Services\Notification;

use App\Models\User;
use App\Services\SMS\SMSService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class NotificationService
{
    protected $smsService;

    public function __construct(SMSService $smsService)
    {
        $this->smsService = $smsService;
    }

    /**
     * Send unified notification (Email + SMS + Push)
     */
    public function send(User $user, string $category, string $subject, string $message, array $data = [])
    {
        $preferences = $user->notification_preferences ?? [];

        $sent = [
            'email' => false,
            'sms' => false,
            'push' => false,
        ];

        // Check if category is enabled for user
        if (!$this->isCategoryEnabled($user, $category)) {
            return $sent;
        }

        // Send Email
        if ($this->shouldSendEmail($user, $category, $preferences)) {
            $sent['email'] = $this->sendEmail($user, $subject, $message, $data);
        }

        // Send SMS
        if ($this->shouldSendSMS($user, $category, $preferences)) {
            $sent['sms'] = $this->sendSMS($user, $message);
        }

        // Send Push Notification
        if ($this->shouldSendPush($user, $category, $preferences)) {
            $sent['push'] = $this->sendPush($user, $subject, $message, $data);
        }

        return $sent;
    }

    /**
     * Send email notification
     */
    protected function sendEmail(User $user, string $subject, string $message, array $data = []): bool
    {
        try {
            Mail::send('emails.notification', [
                'user' => $user,
                'message' => $message,
                'data' => $data,
            ], function ($mail) use ($user, $subject) {
                $mail->to($user->email)
                    ->subject($subject);
            });

            return true;
        } catch (\Exception $e) {
            Log::error('Email notification failed: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'subject' => $subject,
            ]);
            return false;
        }
    }

    /**
     * Send SMS notification
     */
    protected function sendSMS(User $user, string $message): bool
    {
        if (!$user->phone) {
            return false;
        }

        try {
            return $this->smsService->sendSMS($user->phone, $message);
        } catch (\Exception $e) {
            Log::error('SMS notification failed: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'phone' => $user->phone,
            ]);
            return false;
        }
    }

    /**
     * Send push notification
     */
    protected function sendPush(User $user, string $title, string $body, array $data = []): bool
    {
        try {
            // Firebase Cloud Messaging
            $fcmToken = $user->fcm_token;

            if (!$fcmToken) {
                return false;
            }

            $serverKey = config('services.firebase.server_key');

            if (!$serverKey) {
                Log::warning('Firebase server key not configured');
                return false;
            }

            $notification = [
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
            ];

            $payload = [
                'to' => $fcmToken,
                'notification' => $notification,
                'data' => $data,
            ];

            $headers = [
                'Authorization: key=' . $serverKey,
                'Content-Type: application/json',
            ];

            $ch = curl_init('https://fcm.googleapis.com/fcm/send');
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

            $result = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            return $httpCode === 200;
        } catch (\Exception $e) {
            Log::error('Push notification failed: ' . $e->getMessage(), [
                'user_id' => $user->id,
            ]);
            return false;
        }
    }

    /**
     * Check if category is enabled
     */
    protected function isCategoryEnabled(User $user, string $category): bool
    {
        $preferences = $user->notification_preferences ?? [];

        if (!isset($preferences['categories'])) {
            return true; // Default to enabled
        }

        return $preferences['categories'][$category] ?? true;
    }

    /**
     * Should send email
     */
    protected function shouldSendEmail(User $user, string $category, array $preferences): bool
    {
        if (!$user->email) {
            return false;
        }

        return ($preferences['email_enabled'] ?? true);
    }

    /**
     * Should send SMS
     */
    protected function shouldSendSMS(User $user, string $category, array $preferences): bool
    {
        if (!$user->phone) {
            return false;
        }

        return ($preferences['sms_enabled'] ?? false);
    }

    /**
     * Should send push
     */
    protected function shouldSendPush(User $user, string $category, array $preferences): bool
    {
        return ($preferences['push_enabled'] ?? true);
    }

    /**
     * Broadcast system announcement
     */
    public function broadcastAnnouncement(string $subject, string $message, array $options = [])
    {
        $query = User::where('status', 'active');

        if (isset($options['user_type'])) {
            $query->where('account_type', $options['user_type']);
        }

        if (isset($options['subscription_tier'])) {
            $query->whereHas('activeSubscription', function ($q) use ($options) {
                $q->whereHas('plan', function ($q2) use ($options) {
                    $q2->where('tier', $options['subscription_tier']);
                });
            });
        }

        $users = $query->get();
        $sent = 0;

        foreach ($users as $user) {
            $result = $this->send($user, 'system', $subject, $message);
            if ($result['email'] || $result['sms'] || $result['push']) {
                $sent++;
            }
        }

        return [
            'total_users' => $users->count(),
            'sent' => $sent,
        ];
    }

    /**
     * Send notification to multiple users
     */
    public function sendBatch(array $userIds, string $category, string $subject, string $message, array $data = [])
    {
        $users = User::whereIn('id', $userIds)->get();
        $results = [];

        foreach ($users as $user) {
            $results[$user->id] = $this->send($user, $category, $subject, $message, $data);
        }

        return $results;
    }
}
