<?php

namespace App\Services\Payment;

use App\Models\Payment;
use App\Models\Subscription;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use Stripe\Customer;
use Stripe\PaymentIntent;
use Stripe\Webhook;

class StripeService
{
    public function __construct()
    {
        Stripe::setApiKey(config('services.stripe.secret'));
    }

    public function createCheckoutSession($user, $subscriptionPlan, $returnUrl, $cancelUrl)
    {
        try {
            // Create or get Stripe customer
            $customer = $this->getOrCreateCustomer($user);

            $session = Session::create([
                'customer' => $customer->id,
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => strtolower($subscriptionPlan->currency),
                        'product_data' => [
                            'name' => $subscriptionPlan->name,
                            'description' => $subscriptionPlan->description,
                        ],
                        'unit_amount' => $subscriptionPlan->price * 100, // Convert to cents
                        'recurring' => [
                            'interval' => $this->getStripeInterval($subscriptionPlan->duration),
                        ],
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'subscription',
                'success_url' => $returnUrl . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => $cancelUrl,
                'metadata' => [
                    'user_id' => $user->id,
                    'subscription_plan_id' => $subscriptionPlan->id,
                ],
            ]);

            return [
                'success' => true,
                'session_id' => $session->id,
                'checkout_url' => $session->url,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function createPaymentIntent($amount, $currency = 'AED', $metadata = [])
    {
        $paymentIntent = PaymentIntent::create([
            'amount' => $amount * 100, // Convert to cents
            'currency' => strtolower($currency),
            'metadata' => $metadata,
            'automatic_payment_methods' => [
                'enabled' => true,
            ],
        ]);

        return $paymentIntent;
    }

    public function retrievePaymentIntent($paymentIntentId)
    {
        return PaymentIntent::retrieve($paymentIntentId);
    }

    public function refundPayment($paymentIntentId, $amount = null, $reason = null)
    {
        $refundData = ['payment_intent' => $paymentIntentId];

        if ($amount) {
            $refundData['amount'] = $amount; // Already in cents
        }

        if ($reason) {
            $refundData['reason'] = $reason;
        }

        return \Stripe\Refund::create($refundData);
    }

    public function handleWebhook($payload, $signature)
    {
        try {
            $event = Webhook::constructEvent(
                $payload,
                $signature,
                config('services.stripe.webhook_secret')
            );

            switch ($event->type) {
                case 'checkout.session.completed':
                    $this->handleCheckoutSessionCompleted($event->data->object);
                    break;

                case 'payment_intent.succeeded':
                    $this->handlePaymentIntentSucceeded($event->data->object);
                    break;

                case 'payment_intent.payment_failed':
                    $this->handlePaymentIntentFailed($event->data->object);
                    break;

                case 'customer.subscription.created':
                    $this->handleSubscriptionCreated($event->data->object);
                    break;

                case 'customer.subscription.updated':
                    $this->handleSubscriptionUpdated($event->data->object);
                    break;

                case 'customer.subscription.deleted':
                    $this->handleSubscriptionDeleted($event->data->object);
                    break;

                case 'invoice.payment_succeeded':
                    $this->handleInvoicePaymentSucceeded($event->data->object);
                    break;

                case 'invoice.payment_failed':
                    $this->handleInvoicePaymentFailed($event->data->object);
                    break;
            }

            return ['success' => true];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    protected function getOrCreateCustomer($user)
    {
        // Check if user already has Stripe customer ID
        if ($user->stripe_customer_id) {
            try {
                return Customer::retrieve($user->stripe_customer_id);
            } catch (\Exception $e) {
                // Customer not found, create new one
            }
        }

        // Create new customer
        $customer = Customer::create([
            'email' => $user->email,
            'name' => $user->full_name,
            'phone' => $user->phone,
            'metadata' => [
                'user_id' => $user->id,
            ],
        ]);

        // Save customer ID to user
        $user->update(['stripe_customer_id' => $customer->id]);

        return $customer;
    }

    protected function getStripeInterval($duration)
    {
        return match($duration) {
            'monthly' => 'month',
            'quarterly' => 'month', // Handle quarterly as 3 months
            'yearly' => 'year',
            default => 'month',
        };
    }

    protected function handleCheckoutSessionCompleted($session)
    {
        $userId = $session->metadata->user_id ?? null;
        $subscriptionPlanId = $session->metadata->subscription_plan_id ?? null;

        if ($userId && $subscriptionPlanId) {
            $user = \App\Models\User::find($userId);
            $plan = \App\Models\SubscriptionPlan::find($subscriptionPlanId);

            if (!$user || !$plan) {
                \Log::error('User or plan not found in webhook', ['user_id' => $userId, 'plan_id' => $subscriptionPlanId]);
                return;
            }

            // Calculate subscription dates
            $startDate = now();
            $endDate = $this->calculateEndDate($startDate, $plan);

            // Create subscription record
            $subscription = Subscription::create([
                'user_id' => $userId,
                'business_id' => $plan->type === 'business' ? $user->business->id ?? null : null,
                'subscription_plan_id' => $subscriptionPlanId,
                'stripe_subscription_id' => $session->subscription ?? null,
                'status' => 'active',
                'start_date' => $startDate,
                'end_date' => $endDate,
                'trial_end_date' => $plan->trial_days > 0 ? $startDate->copy()->addDays($plan->trial_days) : null,
                'auto_renew' => true,
            ]);

            // Create payment record
            $payment = Payment::create([
                'user_id' => $userId,
                'business_id' => $subscription->business_id,
                'subscription_id' => $subscription->id,
                'payment_method' => 'stripe',
                'transaction_id' => $session->payment_intent ?? $session->id,
                'amount' => $session->amount_total / 100,
                'currency' => strtoupper($session->currency),
                'status' => 'completed',
                'type' => 'subscription',
                'payment_date' => now(),
                'receipt_url' => $session->receipt_url ?? null,
                'metadata' => json_encode($session),
            ]);

            // Update business subscription if applicable
            if ($plan->type === 'business' && $user->business) {
                $user->business->update([
                    'subscription_plan_id' => $subscription->subscription_plan_id,
                    'subscription_status' => 'active',
                    'subscription_start_date' => $subscription->start_date,
                    'subscription_end_date' => $subscription->end_date,
                ]);

                \Log::info('Business subscription updated via webhook', [
                    'business_id' => $user->business->id,
                    'plan_id' => $subscription->subscription_plan_id,
                    'subscription_id' => $subscription->id
                ]);
            }

            // Create notification
            \App\Models\Notification::create([
                'notifiable_type' => 'App\\Models\\User',
                'notifiable_id' => $userId,
                'type' => 'subscription_activated',
                'data' => json_encode([
                    'title' => 'Subscription Activated',
                    'message' => "Your {$plan->name} subscription has been activated successfully.",
                    'subscription_id' => $subscription->id,
                    'plan_name' => $plan->name,
                ])
            ]);

            \Log::info('Subscription created successfully', ['subscription_id' => $subscription->id]);
        }
    }

    protected function handlePaymentIntentSucceeded($paymentIntent)
    {
        // Update payment record
        Payment::where('transaction_id', $paymentIntent->id)
            ->update([
                'status' => 'completed',
                'payment_date' => now(),
            ]);
    }

    protected function handlePaymentIntentFailed($paymentIntent)
    {
        $payment = Payment::where('transaction_id', $paymentIntent->id)->first();

        if ($payment) {
            $payment->update(['status' => 'failed']);

            \App\Models\Notification::create([
                'notifiable_type' => 'App\\Models\\User',
                'notifiable_id' => $payment->user_id,
                'type' => 'payment_failed',
                'data' => json_encode([
                    'title' => 'Payment Failed',
                    'message' => 'Your payment could not be processed. Please try again.',
                    'payment_id' => $payment->id,
                    'amount' => $payment->amount,
                ])
            ]);
        }
    }

    protected function handleSubscriptionCreated($stripeSubscription)
    {
        // Find subscription by stripe_subscription_id
        $subscription = Subscription::where('stripe_subscription_id', $stripeSubscription->id)->first();

        if ($subscription) {
            $subscription->update(['status' => 'active']);
        }
    }

    protected function handleSubscriptionUpdated($stripeSubscription)
    {
        $subscription = Subscription::where('stripe_subscription_id', $stripeSubscription->id)->first();

        if ($subscription) {
            $status = match($stripeSubscription->status) {
                'active' => 'active',
                'canceled' => 'cancelled',
                'past_due' => 'suspended',
                'unpaid' => 'suspended',
                default => $subscription->status,
            };

            $endDate = \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_end);

            $subscription->update([
                'status' => $status,
                'end_date' => $endDate,
            ]);

            // Update business profile if this is a business subscription
            if ($subscription->business_id) {
                $business = \App\Models\Business::find($subscription->business_id);
                if ($business) {
                    $business->update([
                        'subscription_status' => $status,
                        'subscription_end_date' => $endDate,
                    ]);

                    \Log::info('Business subscription status updated via webhook', [
                        'business_id' => $business->id,
                        'new_status' => $status,
                        'end_date' => $endDate
                    ]);
                }
            }
        }
    }

    protected function handleSubscriptionDeleted($stripeSubscription)
    {
        $subscription = Subscription::where('stripe_subscription_id', $stripeSubscription->id)->first();

        if ($subscription) {
            $subscription->update([
                'status' => 'cancelled',
                'cancelled_at' => now(),
                'auto_renew' => false,
            ]);

            // Update business profile if this is a business subscription
            if ($subscription->business_id) {
                $business = \App\Models\Business::find($subscription->business_id);
                if ($business) {
                    $business->update([
                        'subscription_status' => 'cancelled',
                    ]);

                    \Log::info('Business subscription cancelled via webhook', [
                        'business_id' => $business->id,
                        'subscription_id' => $subscription->id
                    ]);
                }
            }

            // Notify user
            \App\Models\Notification::create([
                'notifiable_type' => 'App\\Models\\User',
                'notifiable_id' => $subscription->user_id,
                'type' => 'subscription_cancelled',
                'data' => json_encode([
                    'title' => 'Subscription Cancelled',
                    'message' => 'Your subscription has been cancelled.',
                    'subscription_id' => $subscription->id,
                ])
            ]);
        }
    }

    protected function handleInvoicePaymentSucceeded($invoice)
    {
        // Handle successful recurring payment
        $stripeSubscriptionId = $invoice->subscription ?? null;

        if ($stripeSubscriptionId) {
            $subscription = Subscription::where('stripe_subscription_id', $stripeSubscriptionId)->first();

            if ($subscription) {
                // Create payment record for recurring payment
                Payment::create([
                    'user_id' => $subscription->user_id,
                    'business_id' => $subscription->business_id,
                    'subscription_id' => $subscription->id,
                    'payment_method' => 'stripe',
                    'transaction_id' => $invoice->payment_intent,
                    'amount' => $invoice->amount_paid / 100,
                    'currency' => strtoupper($invoice->currency),
                    'status' => 'completed',
                    'type' => 'subscription',
                    'payment_date' => now(),
                    'receipt_url' => $invoice->invoice_pdf ?? null,
                    'metadata' => json_encode($invoice),
                ]);

                // Extend subscription
                $newEndDate = \Carbon\Carbon::createFromTimestamp($invoice->period_end);
                $subscription->update([
                    'end_date' => $newEndDate,
                    'status' => 'active',
                ]);

                // Update business profile if this is a business subscription
                if ($subscription->business_id) {
                    $business = \App\Models\Business::find($subscription->business_id);
                    if ($business) {
                        $business->update([
                            'subscription_status' => 'active',
                            'subscription_end_date' => $newEndDate,
                        ]);

                        \Log::info('Business subscription renewed via webhook', [
                            'business_id' => $business->id,
                            'new_end_date' => $newEndDate
                        ]);
                    }
                }

                // Notify user
                \App\Models\Notification::create([
                    'notifiable_type' => 'App\\Models\\User',
                    'notifiable_id' => $subscription->user_id,
                    'type' => 'payment_received',
                    'data' => json_encode([
                        'title' => 'Payment Received',
                        'message' => 'Your subscription payment was processed successfully.',
                        'amount' => $invoice->amount_paid / 100,
                        'subscription_id' => $subscription->id,
                    ])
                ]);
            }
        }
    }

    protected function handleInvoicePaymentFailed($invoice)
    {
        $stripeSubscriptionId = $invoice->subscription ?? null;

        if ($stripeSubscriptionId) {
            $subscription = Subscription::where('stripe_subscription_id', $stripeSubscriptionId)->first();

            if ($subscription) {
                // Update subscription status
                $subscription->update(['status' => 'suspended']);

                // Update business profile if this is a business subscription
                if ($subscription->business_id) {
                    $business = \App\Models\Business::find($subscription->business_id);
                    if ($business) {
                        $business->update([
                            'subscription_status' => 'suspended',
                        ]);

                        \Log::warning('Business subscription suspended due to payment failure', [
                            'business_id' => $business->id,
                            'subscription_id' => $subscription->id
                        ]);
                    }
                }

                // Create failed payment record
                Payment::create([
                    'user_id' => $subscription->user_id,
                    'business_id' => $subscription->business_id,
                    'subscription_id' => $subscription->id,
                    'payment_method' => 'stripe',
                    'transaction_id' => $invoice->payment_intent,
                    'amount' => $invoice->amount_due / 100,
                    'currency' => strtoupper($invoice->currency),
                    'status' => 'failed',
                    'type' => 'subscription',
                    'metadata' => json_encode($invoice),
                ]);

                // Notify user
                \App\Models\Notification::create([
                    'notifiable_type' => 'App\\Models\\User',
                    'notifiable_id' => $subscription->user_id,
                    'type' => 'payment_failed',
                    'data' => json_encode([
                        'title' => 'Payment Failed',
                        'message' => 'Your subscription payment failed. Please update your payment method.',
                        'amount' => $invoice->amount_due / 100,
                        'subscription_id' => $subscription->id,
                    ])
                ]);
            }
        }
    }

    protected function calculateEndDate($startDate, $plan)
    {
        return match($plan->duration) {
            'monthly' => $startDate->copy()->addMonth(),
            'quarterly' => $startDate->copy()->addMonths(3),
            'yearly' => $startDate->copy()->addYear(),
            default => $startDate->copy()->addMonth(),
        };
    }

    public function cancelSubscription($stripeSubscriptionId)
    {
        try {
            $subscription = \Stripe\Subscription::retrieve($stripeSubscriptionId);
            $subscription->cancel();

            return ['success' => true];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function refund($paymentIntentId, $amount = null)
    {
        try {
            $refundData = ['payment_intent' => $paymentIntentId];

            if ($amount) {
                $refundData['amount'] = $amount * 100; // Convert to cents
            }

            $refund = \Stripe\Refund::create($refundData);

            return [
                'success' => true,
                'refund_id' => $refund->id,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }
}
