<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('notifications', function (Blueprint $table) {
            $table->id();
            $table->string('type');
            $table->morphs('notifiable'); // creates notifiable_type, notifiable_id + index automatically
            $table->text('data');
            $table->timestamp('read_at')->nullable();
            $table->timestamps();

            // $table->index(['notifiable_type', 'notifiable_id']); // REMOVE THIS LINE
        });

        Schema::create('notification_preferences', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');

            // Channel preferences
            $table->boolean('email_enabled')->default(true);
            $table->boolean('sms_enabled')->default(false);
            $table->boolean('push_enabled')->default(true);

            // Category preferences (9 categories as per requirements)
            $table->boolean('account_notifications')->default(true);
            $table->boolean('offer_new_notifications')->default(true);
            $table->boolean('offer_update_notifications')->default(true);
            $table->boolean('offer_exclusive_notifications')->default(true);
            $table->boolean('redemption_notifications')->default(true);
            $table->boolean('subscription_notifications')->default(true);
            $table->boolean('spotlight_notifications')->default(true);
            $table->boolean('system_notifications')->default(true);
            $table->boolean('marketing_notifications')->default(false);

            $table->timestamps();

            $table->unique('user_id');
        });

        Schema::create('fcm_tokens', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->string('token');
            $table->string('device_type')->nullable(); // ios, android, web
            $table->string('device_id')->nullable();
            $table->timestamp('last_used_at')->nullable();
            $table->timestamps();

            $table->unique(['user_id', 'token']);
            $table->index('user_id');
        });

        Schema::create('notification_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null');
            $table->string('type'); // email, sms, push
            $table->string('category'); // account, offer_new, etc.
            $table->string('recipient'); // email address, phone number, or fcm token
            $table->string('status'); // sent, failed, pending
            $table->text('subject')->nullable();
            $table->text('content')->nullable();
            $table->text('error_message')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamp('sent_at')->nullable();
            $table->timestamps();

            $table->index(['user_id', 'type', 'status']);
            $table->index(['category', 'created_at']);
        });

        Schema::create('notification_templates', function (Blueprint $table) {
            $table->id();
            $table->string('code')->unique(); // unique identifier like 'offer_new', 'redemption_success'
            $table->string('name');
            $table->string('category'); // account, offer_new, offer_update, etc.
            $table->text('description')->nullable();

            // Email template
            $table->string('email_subject')->nullable();
            $table->text('email_body')->nullable();
            $table->boolean('email_enabled')->default(true);

            // SMS template
            $table->text('sms_body')->nullable();
            $table->boolean('sms_enabled')->default(false);

            // Push notification template
            $table->string('push_title')->nullable();
            $table->text('push_body')->nullable();
            $table->boolean('push_enabled')->default(true);

            // Variables available in template (JSON array)
            $table->json('available_variables')->nullable();

            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->index('code');
            $table->index(['category', 'is_active']);
        });

        Schema::create('scheduled_notifications', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->string('template_code');
            $table->string('type'); // email, sms, push
            $table->json('data'); // variables for template
            $table->timestamp('scheduled_at');
            $table->string('status')->default('pending'); // pending, sent, failed, cancelled
            $table->timestamp('sent_at')->nullable();
            $table->text('error_message')->nullable();
            $table->integer('retry_count')->default(0);
            $table->timestamps();

            $table->index(['scheduled_at', 'status']);
            $table->index(['user_id', 'status']);
        });

        Schema::create('push_notification_topics', function (Blueprint $table) {
            $table->id();
            $table->string('topic')->unique();
            $table->string('name');
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        Schema::create('user_notification_topics', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('topic_id')->references('id')->on('push_notification_topics')->onDelete('cascade');
            $table->timestamps();

            $table->unique(['user_id', 'topic_id']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('user_notification_topics');
        Schema::dropIfExists('push_notification_topics');
        Schema::dropIfExists('scheduled_notifications');
        Schema::dropIfExists('notification_templates');
        Schema::dropIfExists('notification_logs');
        Schema::dropIfExists('fcm_tokens');
        Schema::dropIfExists('notification_preferences');
        Schema::dropIfExists('notifications');
    }
};
