<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Add blackout dates to offers
        Schema::table('offers', function (Blueprint $table) {
            $table->json('blackout_dates')->nullable()->after('end_date');
            $table->integer('max_redemptions_per_user')->nullable()->after('redemption_limit');
            $table->integer('max_redemptions_per_user_per_day')->nullable()->after('max_redemptions_per_user');
            $table->integer('daily_redemption_limit')->nullable()->after('max_redemptions_per_user_per_day');
        });

        // Create feedback table
        Schema::create('feedbacks', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->integer('rating')->comment('1-5 stars');
            $table->text('feedback');
            $table->enum('category', ['app', 'features', 'support', 'general'])->default('general');
            $table->enum('status', ['new', 'reviewed', 'archived'])->default('new');
            $table->foreignId('reviewed_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamp('reviewed_at')->nullable();
            $table->timestamps();

            $table->index(['status', 'rating', 'created_at']);
        });

        // Create contact messages table
        Schema::create('contact_messages', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('email');
            $table->string('phone')->nullable();
            $table->string('subject');
            $table->text('message');
            $table->enum('type', ['inquiry', 'support', 'business', 'partnership'])->default('inquiry');
            $table->enum('status', ['new', 'in_progress', 'resolved', 'closed'])->default('new');
            $table->foreignId('assigned_to')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamps();

            $table->index(['status', 'type', 'created_at']);
        });

        // Add notification categories
        Schema::table('notifications', function (Blueprint $table) {
            if (!Schema::hasColumn('notifications', 'category')) {
                $table->enum('category', [
                    'account',
                    'offer_new',
                    'offer_update',
                    'offer_exclusive',
                    'redemption',
                    'subscription',
                    'spotlight',
                    'system',
                    'marketing'
                ])->default('system')->after('notifiable_id');
            }
        });

        // Create offer A/B tests table
        Schema::create('offer_ab_tests', function (Blueprint $table) {
            $table->id();
            $table->foreignId('business_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->string('description')->nullable();
            $table->foreignId('offer_a_id')->constrained('offers')->cascadeOnDelete();
            $table->foreignId('offer_b_id')->constrained('offers')->cascadeOnDelete();
            $table->date('start_date');
            $table->date('end_date');
            $table->integer('traffic_split')->default(50)->comment('Percentage to offer A');
            $table->enum('status', ['draft', 'active', 'paused', 'completed'])->default('draft');
            $table->json('metrics')->nullable();
            $table->foreignId('winner_offer_id')->nullable()->constrained('offers')->nullOnDelete();
            $table->timestamps();

            $table->index(['business_id', 'status', 'start_date']);
        });

        // Create message table for internal communication
        Schema::create('messages', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sender_id')->constrained('users')->cascadeOnDelete();
            $table->foreignId('receiver_id')->constrained('users')->cascadeOnDelete();
            $table->foreignId('business_id')->nullable()->constrained()->nullOnDelete();
            $table->text('message');
            $table->json('attachments')->nullable();
            $table->boolean('is_read')->default(false);
            $table->timestamp('read_at')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->index(['receiver_id', 'is_read', 'created_at']);
            $table->index(['sender_id', 'receiver_id', 'created_at']);
        });

        // Create conversations table
        Schema::create('conversations', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->foreignId('business_id')->nullable()->constrained()->nullOnDelete();
            $table->string('subject')->nullable();
            $table->timestamp('last_message_at')->nullable();
            $table->timestamps();

            $table->index(['user_id', 'business_id', 'last_message_at']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('conversations');
        Schema::dropIfExists('messages');
        Schema::dropIfExists('offer_ab_tests');
        Schema::table('notifications', function (Blueprint $table) {
            if (Schema::hasColumn('notifications', 'category')) {
                $table->dropColumn('category');
            }
        });
        Schema::dropIfExists('contact_messages');
        Schema::dropIfExists('feedbacks');
        Schema::table('offers', function (Blueprint $table) {
            $table->dropColumn([
                'blackout_dates',
                'max_redemptions_per_user',
                'max_redemptions_per_user_per_day',
                'daily_redemption_limit',
            ]);
        });
    }
};
