<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Role;
use App\Models\Permission;

class RolePermissionSeeder extends Seeder
{
    public function run(): void
    {
        // List of permissions
        $permissions = [
            ['name' => 'view_users', 'display_name' => 'View Users', 'description' => 'View user list'],
            ['name' => 'create_users', 'display_name' => 'Create Users', 'description' => 'Create new users'],
            ['name' => 'edit_users', 'display_name' => 'Edit Users', 'description' => 'Edit user details'],
            ['name' => 'delete_users', 'display_name' => 'Delete Users', 'description' => 'Delete users'],
            ['name' => 'view_businesses', 'display_name' => 'View Businesses', 'description' => 'View business list'],
            ['name' => 'create_businesses', 'display_name' => 'Create Businesses', 'description' => 'Create new businesses'],
            ['name' => 'edit_businesses', 'display_name' => 'Edit Businesses', 'description' => 'Edit business details'],
            ['name' => 'delete_businesses', 'display_name' => 'Delete Businesses', 'description' => 'Delete businesses'],
            ['name' => 'view_venues', 'display_name' => 'View Venues', 'description' => 'View venue list'],
            ['name' => 'create_venues', 'display_name' => 'Create Venues', 'description' => 'Create new venues'],
            ['name' => 'edit_venues', 'display_name' => 'Edit Venues', 'description' => 'Edit venue details'],
            ['name' => 'delete_venues', 'display_name' => 'Delete Venues', 'description' => 'Delete venues'],
        ];

        // Create permissions
        foreach ($permissions as $permission) {
            Permission::firstOrCreate(
                ['name' => $permission['name']],
                [
                    'display_name' => $permission['display_name'],
                    'description' => $permission['description'],
                ]
            );
        }

        // Define roles with permission names
        $roles = [
            'super_admin' => [
                'display_name' => 'Super Admin',
                'description' => 'Full access to all system features',
                'permissions' => Permission::pluck('name')->toArray(),
            ],
            'admin' => [
                'display_name' => 'Admin',
                'description' => 'Manage users and businesses',
                'permissions' => [
                    'view_users', 'create_users', 'edit_users', 'delete_users',
                    'view_businesses', 'create_businesses', 'edit_businesses', 'delete_businesses',
                ],
            ],
            'business_owner' => [
                'display_name' => 'Business Owner',
                'description' => 'Manage their own business venues and offers',
                'permissions' => [
                    'view_venues', 'create_venues', 'edit_venues', 'delete_venues',
                ],
            ],
        ];

        // Create roles and assign permissions
        foreach ($roles as $roleName => $data) {
            $role = Role::firstOrCreate(
                ['name' => $roleName],
                [
                    'display_name' => $data['display_name'],
                    'description' => $data['description'],
                ]
            );

            // Get permission IDs based on names
            $permissionIds = Permission::whereIn('name', $data['permissions'])->pluck('id')->toArray();

            // Sync permissions with IDs
            $role->permissions()->sync($permissionIds);
        }

        $this->command->info('✅ Roles & Permissions seeded successfully!');
    }
}
