@extends('layouts.admin')

@section('title', 'Category Management')

@section('content')
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2 class="fw-bold mb-1">Category Management</h2>
        <p class="text-muted mb-0">Organize and manage business categories</p>
    </div>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createCategoryModal">
        <i class="bi bi-plus-lg me-2"></i>Add Category
    </button>
</div>

<!-- Stats -->
<div class="row g-4 mb-4">
    <div class="col-md-3">
        <div class="stat-card">
            <div class="stat-label">Total Categories</div>
            <div class="stat-value" style="font-size: 28px;">{{ $stats['total'] ?? 24 }}</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card">
            <div class="stat-label">Active Categories</div>
            <div class="stat-value" style="font-size: 28px; color: var(--admin-secondary);">{{ $stats['active'] ?? 22 }}</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card">
            <div class="stat-label">Total Businesses</div>
            <div class="stat-value" style="font-size: 28px;">{{ $stats['businesses'] ?? 487 }}</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card">
            <div class="stat-label">Total Offers</div>
            <div class="stat-value" style="font-size: 28px;">{{ $stats['offers'] ?? 1243 }}</div>
        </div>
    </div>
</div>

<!-- Categories Grid -->
<div class="admin-card">
    <div class="admin-card-header">
        <h5 class="mb-0 fw-bold">All Categories</h5>
    </div>
    <div class="admin-card-body">
        <div class="row g-4">
            @forelse($categories ?? [] as $category)
            <div class="col-md-6 col-lg-4">
                <div class="card border h-100">
                    <div class="card-body">
                        <div class="d-flex align-items-start justify-content-between mb-3">
                            <div class="d-flex align-items-center">
                                <div class="rounded d-flex align-items-center justify-content-center me-3"
                                     style="width: 50px; height: 50px; background: {{ $category->color ?? '#e8eaf6' }};">
                                    <i class="bi bi-{{ $category->icon ?? 'tag' }} fs-4" style="color: {{ $category->icon_color ?? '#667eea' }};"></i>
                                </div>
                                <div>
                                    <h6 class="mb-0">{{ $category->name }}</h6>
                                    <small class="text-muted">{{ $category->name_ar ?? '' }}</small>
                                </div>
                            </div>
                            <div class="dropdown">
                                <button class="btn btn-sm btn-light" data-bs-toggle="dropdown">
                                    <i class="bi bi-three-dots-vertical"></i>
                                </button>
                                <ul class="dropdown-menu dropdown-menu-end">
                                    <li><a class="dropdown-item" href="#" onclick="editCategory({{ $category->id }})">
                                        <i class="bi bi-pencil me-2"></i>Edit
                                    </a></li>
                                    <li><a class="dropdown-item" href="#" onclick="toggleCategory({{ $category->id }})">
                                        <i class="bi bi-toggle-on me-2"></i>Toggle Status
                                    </a></li>
                                    <li><hr class="dropdown-divider"></li>
                                    <li><a class="dropdown-item text-danger" href="#" onclick="deleteCategory({{ $category->id }})">
                                        <i class="bi bi-trash me-2"></i>Delete
                                    </a></li>
                                </ul>
                            </div>
                        </div>

                        <p class="text-muted small mb-3">{{ Str::limit($category->description ?? '', 80) }}</p>

                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <small class="text-muted">
                                    <i class="bi bi-building me-1"></i>{{ $category->businesses_count ?? 0 }} businesses
                                </small>
                                <br>
                                <small class="text-muted">
                                    <i class="bi bi-tag me-1"></i>{{ $category->offers_count ?? 0 }} offers
                                </small>
                            </div>
                            <span class="badge {{ $category->is_active ? 'bg-success' : 'bg-secondary' }}">
                                {{ $category->is_active ? 'Active' : 'Inactive' }}
                            </span>
                        </div>
                    </div>
                </div>
            </div>
            @empty
            <div class="col-12">
                <div class="text-center py-5">
                    <i class="bi bi-grid fs-1 text-muted d-block mb-3"></i>
                    <h5>No categories yet</h5>
                    <p class="text-muted">Create your first category to get started</p>
                </div>
            </div>
            @endforelse
        </div>
    </div>
</div>

<!-- Create Category Modal -->
<div class="modal fade" id="createCategoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form action="{{ route('admin.categories.store') }}" method="POST" enctype="multipart/form-data">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">Create New Category</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-12">
                            <label class="form-label">Category Name (English) <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="name" required>
                        </div>

                        <div class="col-12">
                            <label class="form-label">Category Name (Arabic)</label>
                            <input type="text" class="form-control" name="name_ar" dir="rtl">
                        </div>

                        <div class="col-12">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" name="description" rows="3"></textarea>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Icon</label>
                            <select class="form-select" name="icon">
                                <option value="tag">Tag</option>
                                <option value="cup">Food & Dining</option>
                                <option value="cart">Shopping</option>
                                <option value="heart">Health & Beauty</option>
                                <option value="car">Automotive</option>
                                <option value="house">Home & Garden</option>
                                <option value="tv">Electronics</option>
                                <option value="book">Education</option>
                                <option value="airplane">Travel</option>
                                <option value="controller">Entertainment</option>
                            </select>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Display Order</label>
                            <input type="number" class="form-control" name="sort_order" value="0">
                        </div>

                        <div class="col-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="is_active" id="isActive" checked>
                                <label class="form-check-label" for="isActive">
                                    Active
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Category</button>
                </div>
            </form>
        </div>
    </div>
</div>


<!-- Edit Category Modal -->
<div class="modal fade" id="editCategoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="editCategoryForm" method="POST">
                @csrf
                @method('PUT')

                <div class="modal-header">
                    <h5 class="modal-title">Edit Category</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>

                <div class="modal-body">
                    <input type="hidden" id="edit_id">

                    <div class="row g-3">
                        <div class="col-12">
                            <label class="form-label">Category Name (English) *</label>
                            <input type="text" class="form-control" id="edit_name" name="name" required>
                        </div>

                        <div class="col-12">
                            <label class="form-label">Category Name (Arabic)</label>
                            <input type="text" class="form-control" id="edit_name_ar" name="name_ar">
                        </div>

                        <div class="col-12">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" id="edit_description" name="description" rows="3"></textarea>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Icon</label>
                            <select class="form-select" id="edit_icon" name="icon">
                                <option value="tag">Tag</option>
                                <option value="cup">Food & Dining</option>
                                <option value="cart">Shopping</option>
                                <option value="heart">Health & Beauty</option>
                                <option value="car">Automotive</option>
                                <option value="house">Home & Garden</option>
                                <option value="tv">Electronics</option>
                                <option value="book">Education</option>
                                <option value="airplane">Travel</option>
                                <option value="controller">Entertainment</option>
                            </select>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Display Order</label>
                            <input type="number" class="form-control" id="edit_sort_order" name="sort_order" value="0">
                        </div>

                        <div class="col-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="edit_is_active" name="is_active">
                                <label class="form-check-label">Active</label>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Category</button>
                </div>

            </form>
        </div>
    </div>
</div>

@endsection

@push('scripts')
<script>
function editCategory(id) {
    // Open Modal
    let editModal = new bootstrap.Modal(document.getElementById('editCategoryModal'));

    // Fetch category details
    fetch(`/admin/categories/${id}`)
        .then(response => response.json())
        .then(category => {
            // Fill fields
            document.getElementById('edit_id').value = id;
            document.getElementById('edit_name').value = category.name;
            document.getElementById('edit_name_ar').value = category.name_ar ?? '';
            document.getElementById('edit_description').value = category.description ?? '';
            document.getElementById('edit_icon').value = category.icon ?? 'tag';
            document.getElementById('edit_sort_order').value = category.sort_order ?? 0;
            document.getElementById('edit_is_active').checked = category.is_active == 1;

            // Update form action
            document.getElementById('editCategoryForm').action = `/admin/categories/${id}`;

            // Show modal
            editModal.show();
        });
}

function toggleCategory(id) {
    fetch(`/admin/categories/${id}/toggle`, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        }
    });
}

function deleteCategory(id) {
    if (confirm('Are you sure you want to delete this category? This action cannot be undone.')) {
        fetch(`/admin/categories/${id}`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert(data.message || 'Failed to delete category');
            }
        });
    }
}
</script>
@endpush
