@extends('layouts.admin')

@section('title', 'Offer Details - ' . $offer->title)

@section('content')
    <div class="admin-header mb-4">
        <div>
            <h2>{{ $offer->title }}</h2>
            <p class="text-muted mb-0">
                Business: {{ $offer->business->business_name }} | Venue: {{ $offer->venue->name }}
            </p>
        </div>
        <div class="d-flex gap-2">
            @if($offer->approval_status === 'pending')
                <button class="btn btn-success" onclick="approveOffer({{ $offer->id }})">
                    <i class="bi bi-check-circle me-2"></i>Approve
                </button>
                <button class="btn btn-danger" onclick="rejectOffer({{ $offer->id }})">
                    <i class="bi bi-x-circle me-2"></i>Reject
                </button>
            @endif
            <a href="{{ route('admin.offers.index') }}" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left me-2"></i>Back
            </a>
        </div>
    </div>

    <!-- Status & Stats -->
    <div class="row g-4 mb-4">
        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Approval Status</div>
                    @if($offer->approval_status === 'approved')
                        <span class="badge bg-success px-3 py-2">
                        <i class="bi bi-check-circle me-1"></i>Approved
                    </span>
                    @elseif($offer->approval_status === 'pending')
                        <span class="badge bg-warning px-3 py-2">
                        <i class="bi bi-clock me-1"></i>Pending
                    </span>
                    @elseif($offer->approval_status === 'rejected')
                        <span class="badge bg-danger px-3 py-2">
                        <i class="bi bi-x-circle me-1"></i>Rejected
                    </span>
                    @else
                        <span class="badge bg-secondary px-3 py-2">{{ ucfirst($offer->approval_status) }}</span>
                    @endif
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Status</div>
                    @if($offer->status === 'active')
                        <span class="badge bg-success px-3 py-2">Active</span>
                    @elseif($offer->status === 'expired')
                        <span class="badge bg-danger px-3 py-2">Expired</span>
                    @else
                        <span class="badge bg-secondary px-3 py-2">{{ ucfirst($offer->status) }}</span>
                    @endif
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Total Redemptions</div>
                    <div class="fs-4 fw-bold text-success">{{ $offer->redemption_count }}</div>
                    @if($offer->total_redemption_limit)
                        <small class="text-muted">of {{ $offer->total_redemption_limit }}</small>
                    @endif
                </div>
            </div>
        </div>

        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="text-muted small mb-2">Total Views</div>
                    <div class="fs-4 fw-bold text-info">{{ $offer->view_count }}</div>
                </div>
            </div>
        </div>
    </div>

    <div class="row g-4">
        <div class="col-lg-8">
            <!-- Featured Image -->
            @if($offer->image_url)
                <div class="card border-0 shadow-sm mb-4">
                    <img src="{{ Storage::url($offer->image_url) }}"
                         class="card-img-top"
                         alt="{{ $offer->title }}"
                         style="height: 400px; object-fit: cover;">
                </div>
            @endif

            <!-- Offer Details -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Offer Information</h5>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="text-muted small">Title</label>
                            <div class="fw-semibold">{{ $offer->title }}</div>
                        </div>

                        <div class="col-md-6">
                            <label class="text-muted small">Offer Type</label>
                            <div class="fw-semibold">
                                @if($offer->offer_type === 'exclusive')
                                    <span class="badge bg-warning">Exclusive</span>
                                @elseif($offer->offer_type === 'special')
                                    <span class="badge bg-info">Special</span>
                                @else
                                    <span class="badge bg-primary">Regular</span>
                                @endif
                            </div>
                        </div>

                        <div class="col-12">
                            <label class="text-muted small">Description</label>
                            <div>{{ $offer->description }}</div>
                        </div>

                        <div class="col-md-6">
                            <label class="text-muted small">Discount</label>
                            <div class="fs-5 fw-bold text-success">
                                @if($offer->discount_percentage)
                                    {{ $offer->discount_percentage }}% OFF
                                @elseif($offer->discount_amount)
                                    AED {{ number_format($offer->discount_amount) }} OFF
                                @else
                                    -
                                @endif
                            </div>
                        </div>

                        <div class="col-md-6">
                            <label class="text-muted small">Discount Code</label>
                            <div>
                                @if($offer->discount_code)
                                    <code class="fs-6">{{ $offer->discount_code }}</code>
                                @else
                                    -
                                @endif
                            </div>
                        </div>

                        <div class="col-md-6">
                            <label class="text-muted small">Valid From</label>
                            <div>{{ $offer->start_date ? $offer->start_date->format('M d, Y') : '-' }}</div>
                        </div>

                        <div class="col-md-6">
                            <label class="text-muted small">Valid Until</label>
                            <div>{{ $offer->end_date ? $offer->end_date->format('M d, Y') : '-' }}</div>
                        </div>

                        @if($offer->terms_conditions)
                            <div class="col-12">
                                <label class="text-muted small">Terms & Conditions</label>
                                <div class="border rounded p-3 bg-light">
                                    {{ $offer->terms_conditions }}
                                </div>
                            </div>
                        @endif

                        <div class="col-md-6">
                            <label class="text-muted small">Created</label>
                            <div>{{ $offer->created_at->format('M d, Y h:i A') }}</div>
                        </div>

                        <div class="col-md-6">
                            <label class="text-muted small">Last Updated</label>
                            <div>{{ $offer->updated_at->format('M d, Y h:i A') }}</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Redemption Limits -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Redemption Limits</h5>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <div class="border rounded p-3">
                                <div class="text-muted small mb-1">Total Limit</div>
                                <div class="fs-5 fw-bold">
                                    {{ $offer->redemption_limit ?? 'Unlimited' }}
                                </div>
                                <small class="text-muted">Maximum total redemptions</small>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="border rounded p-3">
                                <div class="text-muted small mb-1">Daily Limit</div>
                                <div class="fs-5 fw-bold">
                                    {{ $offer->daily_redemption_limit ?? 'Unlimited' }}
                                </div>
                                <small class="text-muted">Per day redemptions</small>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="border rounded p-3">
                                <div class="text-muted small mb-1">Per User Limit</div>
                                <div class="fs-5 fw-bold">
                                    {{ $offer->max_redemptions_per_user ?? 'Unlimited' }}
                                </div>
                                <small class="text-muted">Total per user</small>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="border rounded p-3">
                                <div class="text-muted small mb-1">Per User Daily</div>
                                <div class="fs-5 fw-bold">
                                    {{ $offer->max_redemptions_per_user_per_day ?? 'Unlimited' }}
                                </div>
                                <small class="text-muted">Daily per user</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Recent Redemptions -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Recent Redemptions</h5>
                </div>
                <div class="card-body p-0">
                    @php
                        $redemptions = $offer->redemptions()->with('user')->latest()->take(10)->get();
                    @endphp

                    @if($redemptions->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="bg-light">
                                <tr>
                                    <th>User</th>
                                    <th>Code</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                </tr>
                                </thead>
                                <tbody>
                                @foreach($redemptions as $redemption)
                                    <tr>
                                        <td>{{ $redemption->user->full_name }}</td>
                                        <td><code>{{ $redemption->redemption_code }}</code></td>
                                        <td>{{ $redemption->created_at->format('M d, Y h:i A') }}</td>
                                        <td>
                                            @if($redemption->status === 'redeemed')
                                                <span class="badge bg-success">Redeemed</span>
                                            @elseif($redemption->status === 'pending')
                                                <span class="badge bg-warning">Pending</span>
                                            @else
                                                <span class="badge bg-secondary">{{ ucfirst($redemption->status) }}</span>
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <div class="text-center py-4 text-muted">
                            No redemptions yet
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Sidebar -->
        <div class="col-lg-4">
            <!-- Business & Venue Info -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Business & Venue</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label class="text-muted small">Business</label>
                        <div class="fw-semibold">{{ $offer->business->business_name }}</div>
                        <a href="{{ route('admin.businesses.show', $offer->business_id) }}"
                           class="btn btn-sm btn-outline-primary mt-2 w-100">
                            <i class="bi bi-building me-2"></i>View Business
                        </a>
                    </div>

                    <div class="mb-0">
                        <label class="text-muted small">Venue</label>
                        <div class="fw-semibold">{{ $offer->venue->name }}</div>
                        <div class="text-muted small">{{ $offer->venue->city }}, {{ $offer->venue->area }}</div>
                        <a href="{{ route('admin.venues.show', $offer->venue_id) }}"
                           class="btn btn-sm btn-outline-primary mt-2 w-100">
                            <i class="bi bi-geo-alt me-2"></i>View Venue
                        </a>
                    </div>
                </div>
            </div>

            <!-- Performance Metrics -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Performance</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <div class="d-flex justify-content-between mb-1">
                            <span class="text-muted small">Conversion Rate</span>
                            <span class="fw-semibold">
                            @if($offer->view_count > 0)
                                    {{ number_format(($offer->redemption_count / $offer->view_count) * 100, 1) }}%
                                @else
                                    0%
                                @endif
                        </span>
                        </div>
                        <div class="progress" style="height: 8px;">
                            <div class="progress-bar bg-success"
                                 style="width: {{ $offer->view_count > 0 ? ($offer->redemption_count / $offer->view_count) * 100 : 0 }}%">
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <div class="d-flex justify-content-between mb-1">
                            <span class="text-muted small">Redemption Progress</span>
                            <span class="fw-semibold">
                            {{ $offer->redemption_count }}
                                @if($offer->total_redemption_limit)
                                    / {{ $offer->total_redemption_limit }}
                                @endif
                        </span>
                        </div>
                        @if($offer->total_redemption_limit)
                            <div class="progress" style="height: 8px;">
                                <div class="progress-bar bg-primary"
                                     style="width: {{ ($offer->redemption_count / $offer->total_redemption_limit) * 100 }}%">
                                </div>
                            </div>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Admin Actions -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Admin Actions</h5>
                </div>
                <div class="card-body">
                    @if($offer->approval_status === 'pending')
                        <button class="btn btn-success w-100 mb-2" onclick="approveOffer({{ $offer->id }})">
                            <i class="bi bi-check-circle me-2"></i>Approve Offer
                        </button>
                        <button class="btn btn-danger w-100 mb-2" onclick="rejectOffer({{ $offer->id }})">
                            <i class="bi bi-x-circle me-2"></i>Reject Offer
                        </button>
                    @elseif($offer->approval_status === 'approved')
                        <button class="btn btn-warning w-100 mb-2" onclick="pauseOffer({{ $offer->id }})">
                            <i class="bi bi-pause-circle me-2"></i>Pause Offer
                        </button>
                    @endif

                    <button class="btn btn-outline-danger w-100" onclick="deleteOffer({{ $offer->id }})">
                        <i class="bi bi-trash me-2"></i>Delete Offer
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
        function approveOffer(id) {
            if (confirm('Approve this offer?')) {
                fetch(`/admin/offers/${id}/approve`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        } else {
                            alert(data.message || 'Error approving offer');
                        }
                    });
            }
        }

        function rejectOffer(id) {
            const reason = prompt('Enter rejection reason:');
            if (reason) {
                fetch(`/admin/offers/${id}/reject`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ reason: reason })
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        } else {
                            alert(data.message || 'Error rejecting offer');
                        }
                    });
            }
        }

        function pauseOffer(id) {
            if (confirm('Pause this offer?')) {
                fetch(`/admin/offers/${id}/pause`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        } else {
                            alert(data.message || 'Error pausing offer');
                        }
                    });
            }
        }

        function deleteOffer(id) {
            if (confirm('Are you sure you want to delete this offer? This action cannot be undone.')) {
                fetch(`/admin/offers/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            window.location.href = '/admin/offers';
                        } else {
                            alert(data.message || 'Error deleting offer');
                        }
                    });
            }
        }
    </script>
@endpush
